/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#ifdef WIN32
    #include <windows.h>
#else
    #define MB_ICONQUESTION     1
    #define MB_ICONINFORMATION  0
    #define MB_OKCANCEL         1
    #define MB_YESNO            1
    #define MB_TASKMODAL        0
    #define MB_ICONERROR        2
    #define MB_OK               0
    #define IDYES               'y'
    #define IDNO                'n'
    #define IDCANCEL            'n'
    int MessageBox(int hWnd, char *lpText, char *lpCaption, int uType) {
        char    ans[16];
        printf("\n%s%s\n", (uType & 2) ? "Error: " : "", lpText);
        if(uType & 1) {
            printf("- type 'y' for continuing or any other key for quitting\n");
            fgets(ans, sizeof(ans), stdin);
            if((ans[0] != 'y') && (ans[0] != 'Y')) return(IDNO);
        }
        return(IDYES);
    }
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER     "Ventrilo <= 3.0.2 NULL pointer fix 0.1 (any version and platform)"
#define INFO    VER "\n" \
                "by Luigi Auriemma\n" \
                "e-mail: aluigi@autistici.org\n" \
                "web:    aluigi.org\n" \
                "\n" \
                "This unofficial \"universal\" patch is able to fix the problem described\n" \
                "in the following advisory\n" \
                "\n" \
                "  http://aluigi.org/adv/ventrilobotomy-adv.txt\n" \
                "\n" \
                "The work-around is very simple: it disables the message and all the rest of\n" \
                "the operations made by the server if a client uses a version different than\n" \
                "its one avoiding the NULL pointer.\n" \
                "Note that the binaries for SPARC and Mac OSX are not supported"
#define SUCCESS "The file has been successfully patched!\n" \
                "the backup file is available in the same folder with the .backup extension"
#define FIND1   "Protocol error. Server is running version"
#define FIND2   "Incompatible version. Server is running version"

#ifndef MAX_PATH
    #define MAX_PATH    4096
#endif



typedef struct {
    u8      e_ident[16];
    u16     e_type;
    u16     e_machine;
    u32     e_version;
    u32     e_entry;
    u32     e_phoff;
    u32     e_shoff;
    u32     e_flags;
    u16     e_ehsize;
    u16     e_phentsize;
    u16     e_phnum;
    u16     e_shentsize;
    u16     e_shnum;
    u16     e_shstrndx;
} elf_header_t;

typedef struct {
    u32     sh_name;
    u32     sh_type;
    u32     sh_flags;
    u32     sh_addr;     
    u32     sh_offset;
    u32     sh_size;
    u32     sh_link;
    u32     sh_info;
    u32     sh_addralign;
    u32     sh_entsize;
} elf_section_t;

typedef struct {
    u32     rva;
    u32     offset;
    u32     size;
} section_t;



section_t   *section    = NULL;
u32     filememsz       = 0,
        imagebase       = 0,
        sections        = 0;
u8      *filemem        = NULL;



void std_err(u8 *err) {
    if(!err) err = strerror(errno);
    MessageBox(0, err, VER, MB_ICONERROR | MB_OK | MB_TASKMODAL);
    if(section) free(section);
    if(filemem) free(filemem);
    exit(1);
}



u32 file2rva(u32 file) {
    int     i;

    if(!sections) return(file);
    for(i = 0; i < sections; i++) {
        if((file >= section[i].offset) && (file < (section[i].offset + section[i].size))) {
            if((i < (sections - 1)) && ((section[i].offset + section[i].size) > section[i + 1].offset)) continue;   // ELF fix
            return(imagebase + section[i].rva + file - section[i].offset);
        }
    }
    return(-1);
}



int parse_PE(void) {
#ifdef WIN32
    IMAGE_DOS_HEADER        *doshdr;
    IMAGE_NT_HEADERS        *nthdr;
    IMAGE_SECTION_HEADER    *sechdr;
    int     i;
    u8      *p;

    p = filemem;
    doshdr  = (IMAGE_DOS_HEADER *)p;    p += sizeof(IMAGE_DOS_HEADER);
                                        p += doshdr->e_lfanew - sizeof(IMAGE_DOS_HEADER);
    nthdr   = (IMAGE_NT_HEADERS *)p;    p += sizeof(IMAGE_NT_HEADERS);

    if((doshdr->e_magic != IMAGE_DOS_SIGNATURE) || (nthdr->Signature != IMAGE_NT_SIGNATURE) || (nthdr->OptionalHeader.Magic != IMAGE_NT_OPTIONAL_HDR_MAGIC)) {
        return(-1);
    }
    imagebase = nthdr->OptionalHeader.ImageBase;
    sechdr = (IMAGE_SECTION_HEADER *)p;

    sections = nthdr->FileHeader.NumberOfSections;
    section = malloc(sizeof(section_t) * sections);
    if(!section) std_err(NULL);

    for(i = 0; i < sections; i++) {
        section[i].rva    = sechdr[i].VirtualAddress;
        section[i].offset = sechdr[i].PointerToRawData;
        section[i].size   = sechdr[i].SizeOfRawData;
    }
    return(0);
#else
    return(-1);
#endif
}



int parse_ELF(void) {
    elf_header_t    *elf_header;
    elf_section_t   *elf_section;
    int     i;
    u8      *p;

    p = filemem;
    elf_header = (elf_header_t *)p;     p += sizeof(elf_header_t);
    if(memcmp(elf_header->e_ident, "\x7f""ELF", 4)) return(-1);
    if(elf_header->e_ident[4] != 1) return(-1); // only 32 bit supported
    if(elf_header->e_ident[5] != 1) return(-1); // only little endian

    sections = elf_header->e_shnum;
    section = malloc(sizeof(section_t) * sections);
    if(!section) std_err(NULL);

    elf_section = (elf_section_t *)(filemem + elf_header->e_shoff);
    for(i = 0; i < sections; i++) {
        section[i].rva    = elf_section[i].sh_addr;
        section[i].offset = elf_section[i].sh_offset;
        section[i].size   = elf_section[i].sh_size;
    }
    return(0);
}



void load_file(u8 *fname) {
    struct stat xstat;
    FILE    *fd;

    fd = fopen(fname, "rb");
    if(!fd) std_err(NULL);
    fstat(fileno(fd), &xstat);
    filemem = malloc(xstat.st_size);
    if(!filemem) std_err(NULL);
    filememsz = fread(filemem, 1, xstat.st_size, fd);
    fclose(fd);

    if(parse_PE() < 0) {
        if(parse_ELF() < 0) {
            std_err("unsupported input file format");
        }
    }
}



void write_file(u8 *fname, int check) {
    FILE    *fd;
    u8      buff[MAX_PATH + 10 + 64];

    if(check) {
        fd = fopen(fname, "rb");
        if(fd) {
            fclose(fd);
            sprintf(buff, "the file %s already exists\ndo you want to overwrite it?", fname);
            if(MessageBox(0, buff, VER, MB_ICONQUESTION | MB_YESNO | MB_TASKMODAL) == IDNO) exit(1);
        }
    }
    fd = fopen(fname, "wb");
    if(!fd) std_err(NULL);
    fwrite(filemem, 1, filememsz, fd);
    fclose(fd);
}



void get_file(u8 *fname) {
#ifdef WIN32
    OPENFILENAME    ofn;
    static const u8 filter[] =
                    "(ventrilo_srv*)\0" "ventrilo_srv*\0"
                    "(*.*)\0"   "*.*\0"
                    "\0\0";

    fname[0] = 0;
    memset(&ofn, 0, sizeof(ofn));
    ofn.lStructSize     = sizeof(ofn);
    ofn.lpstrFilter     = filter;
    ofn.nFilterIndex    = 1;
    ofn.lpstrFile       = fname;
    ofn.nMaxFile        = MAX_PATH;
    ofn.lpstrTitle      = "Select the Ventrilo server executable";
    ofn.Flags           = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_LONGNAMES | OFN_EXPLORER | OFN_HIDEREADONLY;
    if(!GetOpenFileName(&ofn)) exit(1);
#else
    printf("\n"
        "Usage: ./ventrilobotomyfix <ventrilo_srv>\n");
    exit(1);
#endif
}



u32 find_bytes(int offset, u8 *data, int datalen) {
    u8      *p,
            *l;

    for(p = filemem + offset, l = filemem + filememsz - datalen; p <= l; p++) {
        if(!memcmp(p, data, datalen)) return(p - filemem);
    }
    return(-1);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        data[i] = (num >> (i << 3)) & 0xff;
    }
    return(bytes);
}



int main(int argc, char *argv[]) {
    int     pattern,
            jmp;
    u8      filename[MAX_PATH],
            filenamebck[MAX_PATH + 10],
            buff[256];

    MessageBox(0, INFO, VER, MB_ICONINFORMATION | MB_OK | MB_TASKMODAL);

    if((argc < 2) || (argv[1][0] == '-') || (argv[1][0] == '/')) {
        get_file(filename);
    } else {
        sprintf(filename, "%.*s", sizeof(filename), argv[1]);
    }
    if(!filename[0]) exit(1);
    load_file(filename);

    pattern = find_bytes(0, FIND1, sizeof(FIND1) - 1);
    if(pattern < 0) {
        pattern = find_bytes(0, FIND2, sizeof(FIND2) - 1);
        if(pattern < 0) std_err("the first pattern has not been found");
    }
    pattern = file2rva(pattern);

    putxx(buff, pattern, 32);
    for(jmp = 0;; jmp++) {
        jmp = find_bytes(jmp, buff, 4);
        if(jmp < 0) std_err("the second pattern has not been found");
        if((filemem[jmp - 7] == 0x0f) && (filemem[jmp - 6] == 0x84)) break;
    }
    jmp -= 7;

    sprintf(buff,
        "The pattern has been found and the work-around will be placed at offset:\n"
        "\n"
        "  %08x (RVA %08x)\n"
        "\n"
        "Now I do the backup and the patching of the file\n"
        "Click on Cancel to quit\n", jmp, file2rva(jmp));
    if(MessageBox(0, buff, VER, MB_ICONQUESTION | MB_OKCANCEL | MB_TASKMODAL) == IDCANCEL) goto quit;
    sprintf(filenamebck, "%s.backup", filename);
    write_file(filenamebck, 1);

    filemem[jmp]     = 0x90;
    filemem[jmp + 1] = 0xe9;
    write_file(filename, 0);

    MessageBox(0, SUCCESS, VER, MB_ICONINFORMATION | MB_OK | MB_TASKMODAL);

quit:
    if(section) free(section);
    if(filemem) free(filemem);
    return(0);
}


